'use client'

import { useState, useEffect } from 'react'
import { useRouter } from 'next/navigation'
import Link from 'next/link'
import CartIcon from '../components/CartIcon'

// Convert EUR to BGN and round to nearest 0.10
const convertToBGN = (eur: number | null | undefined): string => {
  if (!eur || eur === 0) return '0.00'
  const bgn = eur * 1.95583
  // Round to nearest 0.10 (e.g., 58.68 -> 58.70, 58.86 -> 58.90)
  const rounded = Math.round(bgn * 10) / 10
  return rounded.toFixed(2)
}

interface CartItem {
  productId: number
  name: string
  price: number
  size: string | null
  color: string | null
  image?: string
}

interface Product {
  id: number
  name: string
  images: string[]
}

const deliveryOptions = [
  { id: 'econt-office', name: 'Офис на Еконт' },
  { id: 'speedy-office', name: 'Офис на Спиди' },
  { id: 'econt-address', name: 'Еконт до Адрес' },
  { id: 'speedy-address', name: 'Спиди до Адрес' },
]

export default function CheckoutPage() {
  const router = useRouter()
  const [cart, setCart] = useState<CartItem[]>([])
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [submitError, setSubmitError] = useState<string | null>(null)
  const [formData, setFormData] = useState({
    name: '',
    email: '',
    phone: '',
    deliveryOption: '',
    deliveryDetails: '',
    comment: '',
  })

  useEffect(() => {
    if (typeof window !== 'undefined') {
      const cartData: CartItem[] = JSON.parse(localStorage.getItem('cart') || '[]')
      
      // Fetch product images for cart items
      fetch('/api/products')
        .then(res => res.json())
        .then((products: Product[]) => {
          const cartWithImages = cartData.map(item => {
            const product = products.find(p => p.id === item.productId)
            if (product && product.images && product.images.length > 0) {
              return {
                ...item,
                image: product.images[0]
              }
            }
            return {
              ...item,
              image: null
            }
          })
          setCart(cartWithImages)
        })
        .catch(() => {
          // If API fails, just use cart data without images
          setCart(cartData)
        })
    }
  }, [])

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target
    setFormData(prev => ({
      ...prev,
      [name]: value,
      ...(name === 'deliveryOption' ? { deliveryDetails: '' } : {}),
    }))
  }

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setSubmitError(null)
    
    if (!formData.name || !formData.email || !formData.phone || !formData.deliveryOption) {
      setSubmitError('Моля, попълнете всички задължителни полета!')
      return
    }

    if (formData.deliveryOption && !formData.deliveryDetails) {
      setSubmitError('Моля, попълнете детайлите за доставка!')
      return
    }

    if (cart.length === 0) {
      setSubmitError('Вашата количка е празна!')
      return
    }

    setIsSubmitting(true)

    const subtotal = cart.reduce((sum, item) => sum + (item.price || 0), 0)
    const deliveryCost = subtotal >= 35 ? 0 : 4
    const total = subtotal + deliveryCost

    // Format cart items for email
    const cartItemsText = cart.map((item, index) => 
      `${index + 1}. ${item.name} - Размер: ${item.size || 'N/A'}, Цвят: ${item.color || 'N/A'} - €${item.price ? item.price.toFixed(2) : '0.00'}`
    ).join('\n')

    // Get delivery option name
    const deliveryOptionName = deliveryOptions.find(opt => opt.id === formData.deliveryOption)?.name || formData.deliveryOption

    // Prepare Formspree data
    const formspreeData = {
      name: formData.name,
      email: formData.email,
      phone: formData.phone,
      deliveryOption: deliveryOptionName,
      deliveryDetails: formData.deliveryDetails,
      comment: formData.comment || '(Няма коментар)',
      cartItems: cartItemsText,
      subtotal: `€${subtotal.toFixed(2)} (${convertToBGN(subtotal)} лв)`,
      deliveryCost: deliveryCost === 0 ? 'Безплатно' : `€${deliveryCost.toFixed(2)} (${convertToBGN(deliveryCost)} лв)`,
      total: `€${total.toFixed(2)} (${convertToBGN(total)} лв)`,
      _subject: `Нова поръчка от ${formData.name} - seksi.bg`,
    }

    try {
      // Replace YOUR_FORM_ID with your actual Formspree form ID
      // You can get this from https://formspree.io/forms
      const formspreeEndpoint = process.env.NEXT_PUBLIC_FORMSPREE_ENDPOINT || 'https://formspree.io/f/YOUR_FORM_ID'
      
      const response = await fetch(formspreeEndpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Accept': 'application/json',
        },
        body: JSON.stringify(formspreeData),
      })

      if (response.ok) {
        // Clear cart and redirect
        if (typeof window !== 'undefined') {
          localStorage.removeItem('cart')
          window.dispatchEvent(new Event('cartUpdated'))
        }
        router.push('/?order=success')
      } else {
        const errorData = await response.json()
        throw new Error(errorData.error || 'Грешка при изпращане на поръчката')
      }
    } catch (error) {
      console.error('Error submitting form:', error)
      setSubmitError('Грешка при изпращане на поръчката. Моля, опитайте отново или се свържете с нас директно.')
      setIsSubmitting(false)
    }
  }

  const subtotal = cart.reduce((sum, item) => sum + (item.price || 0), 0)
  const deliveryCost = subtotal >= 35 ? 0 : 4
  const total = subtotal + deliveryCost

  const needsDeliveryDetails = formData.deliveryOption && formData.deliveryOption !== ''

  return (
    <div className="min-h-screen bg-[#F5F5F5]">
      {/* Announcement Bar */}
      <div className="bg-black text-white text-center py-2 text-sm font-medium">
        БЕЗПЛАТНА ДОСТАВКА НАД 35€
      </div>

      {/* Header */}
      <header className="bg-white">
        <div className="max-w-4xl mx-auto px-4 py-4">
          <div className="flex items-center justify-between">
            <Link href="/" className="text-xl font-bold text-dark-grey">
              ← Назад
            </Link>
            <CartIcon />
          </div>
        </div>
      </header>

      <div className="max-w-4xl mx-auto px-4 py-6">
        <h1 className="text-2xl font-bold text-dark-grey mb-6">Поръчка</h1>

        {cart.length === 0 && (
          <div className="mb-8 p-6 bg-white rounded-lg text-center">
            <p className="text-medium-grey mb-4">Вашата количка е празна.</p>
            <Link href="/" className="text-dark-grey font-medium hover:text-medium-grey">
              Продължете с пазаруването →
            </Link>
          </div>
        )}

        {/* Cart Summary */}
        {cart.length > 0 && (
          <div className="mb-6 p-4 bg-white rounded-lg">
            <h2 className="text-lg font-semibold text-dark-grey mb-4">Вашата поръчка</h2>
            <div className="space-y-4 mb-4">
              {cart.map((item, index) => (
                <div key={index} className="flex gap-3 items-start">
                  {/* Product Image */}
                  <div className="w-20 h-20 flex-shrink-0 rounded-lg overflow-hidden bg-gray-100">
                    {item.image ? (
                      <img
                        src={`/images/products/${item.image}`}
                        alt={item.name}
                        className="w-full h-full object-cover"
                        onError={(e) => {
                          e.currentTarget.src = '/api/placeholder/300/400'
                        }}
                      />
                    ) : (
                      <div className="w-full h-full bg-gradient-to-br from-gray-200 to-gray-300 flex items-center justify-center">
                        <span className="text-medium-grey text-xs">Няма</span>
                      </div>
                    )}
                  </div>
                  {/* Product Details */}
                  <div className="flex-1 min-w-0">
                    <h3 className="text-sm font-medium text-dark-grey mb-1">{item.name}</h3>
                    <div className="text-xs text-medium-grey space-y-1">
                      {item.size && <p>Размер: {item.size}</p>}
                      {item.color && <p>Цвят: {item.color}</p>}
                    </div>
                    <p className="text-sm font-semibold text-dark-grey mt-2">€{item.price ? item.price.toFixed(0) : '0'} ({convertToBGN(item.price)} лв)</p>
                  </div>
                </div>
              ))}
            </div>
            <div className="pt-4 border-t border-gray-200 space-y-2">
              <div className="flex justify-between text-sm text-medium-grey">
                <span>Междинна сума:</span>
                <span>€{subtotal.toFixed(0)} ({convertToBGN(subtotal)} лв)</span>
              </div>
              <div className="flex justify-between text-sm text-medium-grey">
                <span>Доставка:</span>
                <span>
                  {deliveryCost === 0 ? (
                    <span className="text-green-600 font-medium">Безплатно</span>
                  ) : (
                    <>€{deliveryCost.toFixed(0)} ({convertToBGN(deliveryCost)} лв)</>
                  )}
                </span>
              </div>
              <div className="pt-2 border-t border-gray-200 flex justify-between text-lg font-semibold text-dark-grey">
                <span>Общо:</span>
                <span>€{total.toFixed(0)} ({convertToBGN(total)} лв)</span>
              </div>
            </div>
          </div>
        )}

        {/* Checkout Form */}
        {submitError && (
          <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg">
            <p className="text-red-700 text-sm">{submitError}</p>
          </div>
        )}
        <form onSubmit={handleSubmit} className="space-y-6">
          {/* Name */}
          <div>
            <label htmlFor="name" className="block text-sm font-medium text-dark-grey mb-2">
              Име *
            </label>
            <input
              type="text"
              id="name"
              name="name"
              required
              value={formData.name}
              onChange={handleInputChange}
              className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent"
            />
          </div>

          {/* Email */}
          <div>
            <label htmlFor="email" className="block text-sm font-medium text-dark-grey mb-2">
              Имейл *
            </label>
            <input
              type="email"
              id="email"
              name="email"
              required
              value={formData.email}
              onChange={handleInputChange}
              className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent"
            />
          </div>

          {/* Phone */}
          <div>
            <label htmlFor="phone" className="block text-sm font-medium text-dark-grey mb-2">
              Телефон *
            </label>
            <input
              type="tel"
              id="phone"
              name="phone"
              required
              value={formData.phone}
              onChange={handleInputChange}
              className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent"
            />
          </div>

          {/* Delivery Option */}
          <div>
            <label htmlFor="deliveryOption" className="block text-sm font-medium text-dark-grey mb-2">
              Начин на доставка *
            </label>
            <select
              id="deliveryOption"
              name="deliveryOption"
              required
              value={formData.deliveryOption}
              onChange={handleInputChange}
              className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent"
            >
              <option value="">Изберете начин на доставка</option>
              {deliveryOptions.map((option) => (
                <option key={option.id} value={option.id}>
                  {option.name}
                </option>
              ))}
            </select>
          </div>

          {/* Delivery Details */}
          {needsDeliveryDetails && (
            <div>
              <label htmlFor="deliveryDetails" className="block text-sm font-medium text-dark-grey mb-2">
                {formData.deliveryOption.includes('office')
                  ? 'Име на офис *'
                  : 'Адрес за доставка *'}
              </label>
              <input
                type="text"
                id="deliveryDetails"
                name="deliveryDetails"
                required
                value={formData.deliveryDetails}
                onChange={handleInputChange}
                placeholder={
                  formData.deliveryOption.includes('office')
                    ? 'Например: София, офис 123'
                    : 'Например: ул. Примерна 1, София'
                }
                className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent"
              />
            </div>
          )}

          {/* Order Comment */}
          <div>
            <label htmlFor="comment" className="block text-sm font-medium text-dark-grey mb-2">
              Коментар към поръчката
            </label>
            <textarea
              id="comment"
              name="comment"
              rows={4}
              value={formData.comment}
              onChange={handleInputChange}
              placeholder="Добавете коментар или специални инструкции (по избор)..."
              className="w-full px-4 py-3 border border-gray-200 rounded-lg bg-white focus:ring-2 focus:ring-dark-grey focus:border-transparent resize-none"
            />
          </div>

          {/* Submit Button */}
          <button
            type="submit"
            disabled={isSubmitting}
            className={`w-full px-6 py-4 bg-dark-grey text-white font-semibold rounded-lg transition ${
              isSubmitting 
                ? 'opacity-50 cursor-not-allowed' 
                : 'hover:bg-opacity-90'
            }`}
          >
            {isSubmitting ? 'Изпращане...' : 'Изпрати поръчката'}
          </button>
        </form>
      </div>

      {/* Footer */}
      <footer className="bg-white border-t border-gray-200 mt-12">
        <div className="max-w-4xl mx-auto px-4 py-8">
          <div className="grid grid-cols-2 gap-6 mb-6">
            <div>
              <h3 className="text-sm font-semibold text-dark-grey mb-3">За нас</h3>
              <ul className="space-y-2">
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    За нас
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Контакти
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Доставка и връщане
                  </a>
                </li>
              </ul>
            </div>
            <div>
              <h3 className="text-sm font-semibold text-dark-grey mb-3">Помощ</h3>
              <ul className="space-y-2">
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Често задавани въпроси
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Размери
                  </a>
                </li>
                <li>
                  <a href="#" className="text-sm text-medium-grey hover:text-dark-grey transition">
                    Условия за ползване
                  </a>
                </li>
              </ul>
            </div>
          </div>
          <div className="pt-6 border-t border-gray-200 text-center">
            <p className="text-xs text-medium-grey">
              © {new Date().getFullYear()} seksi.bg. Всички права запазени.
            </p>
          </div>
        </div>
      </footer>
    </div>
  )
}

